'use strict';

/* --------------------------------------------------------------
 resize.js 2016-05-12
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2016 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * Resize Layout Controller
 *
 * During the admin layout lifecycle there are events that will change the size of the document (not the window!)
 * and the layout must react to them. This controller will make sure that the layout will remain stable after such
 * changes are marked with the "data-resize-layout" attribute as in the following example.
 *
 * ```html
 * <!-- DataTable Instance -->
 * <table data-gx-widget="datatable" data-resize-layout="draw.dt">
 *   ...
 * </table>
 * ```
 *
 * After a table draw is performed, it is possible that there will be more rows to be displayed and thus the
 * #main-content element gets bigger. Once the datatable "draw.dt" event is executed this module will make
 * sure that the layout remains solid.
 *
 * The event must bubble up to the container this module is bound.
 *
 * ### Dynamic Elements
 *
 * It is possible that during the page lifecycle there will be dynamic elements that will need to register
 * an the "resize-layout" event. In this case apply the "data-resize-layout" attribute in the dynamic
 * element and trigger the "resize:bind" event from that element. The event must bubble up to the layout
 * container which will then register the dynamic elements.
 */
gx.controllers.module('resize', [], function (data) {

    'use strict';

    // ------------------------------------------------------------------------
    // VARIABLES
    // ------------------------------------------------------------------------

    /**
     * Marks event listeners.
     *
     * @type {string}
     */

    var ATTRIBUTE_NAME = 'data-resize-layout';

    /**
     * Module Selector
     *
     * @type {jQuery}
     */
    var $this = $(this);

    /**
     * Module Instance
     *
     * @type {Object}
     */
    var module = {};

    /**
     * Main Header Selector
     *
     * @type {jQuery}
     */
    var $mainHeader = $('#main-header');

    /**
     * Main Menu Selector
     *
     * @type {jQuery}
     */
    var $mainMenu = $('#main-menu');

    /**
     * Main Footer Selector
     *
     * @type {jQuery}
     */
    var $mainFooter = $('#main-footer');

    /**
     * Main Footer Info
     *
     * @type {jQuery}
     */
    var $mainFooterInfo = $mainFooter.find('.info');

    /**
     * Main Footer Copyright
     *
     * @type {jQuery}
     */
    var $mainFooterCopyright = $mainFooter.find('.copyright');

    // ------------------------------------------------------------------------
    // FUNCTIONS
    // ------------------------------------------------------------------------

    /**
     * Bind resize events.
     */
    function _bindResizeEvents() {
        $this.find('[' + ATTRIBUTE_NAME + ']').each(function () {
            var event = $(this).attr(ATTRIBUTE_NAME);
            $(this).removeAttr(ATTRIBUTE_NAME).on(event, _updateLayoutComponents);
        });
    }

    /**
     * Give initial min height to main menu.
     */
    function _updateLayoutComponents() {
        var mainMenuHeight = window.innerHeight - $mainHeader.outerHeight() - $mainFooter.outerHeight() + $mainFooterCopyright.outerHeight();
        $mainMenu.css('min-height', mainMenuHeight);
        _setFooterInfoPosition();
    }

    /**
     * Calculate the correct footer info position.
     */
    function _setFooterInfoPosition() {
        if ($(document).scrollTop() + window.innerHeight - $mainFooterInfo.outerHeight() <= $mainFooter.offset().top) {
            $mainFooter.addClass('fixed');
        } else if ($mainFooterInfo.offset().top + $mainFooterInfo.height() >= $mainFooter.offset().top) {
            $mainFooter.removeClass('fixed');
        }
    }

    // ------------------------------------------------------------------------
    // INITIALIZATION
    // ------------------------------------------------------------------------

    module.init = function (done) {
        $(window).on('resize', _updateLayoutComponents).on('JSENGINE_INIT_FINISHED', _updateLayoutComponents).on('scroll', _setFooterInfoPosition).on('register:bind', _bindResizeEvents);

        _bindResizeEvents();

        done();
    };

    return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
